const fs = require('fs');
const path = require('path');
const https = require('https');
const { URL } = require('url');

const {
  loadRegistryFromFile,
  normalizeRemoteMetadata
} = require('./registry');

const RETRY_DELAYS_MS = [15_000, 30_000, 60_000, 120_000, 300_000];

function requestJson(urlString, { timeoutMs, etag } = {}) {
  return new Promise((resolve, reject) => {
    let parsed;
    try {
      parsed = new URL(urlString);
    } catch (error) {
      reject(error);
      return;
    }

    const options = {
      protocol: parsed.protocol,
      hostname: parsed.hostname,
      port: parsed.port || (parsed.protocol === 'https:' ? 443 : 80),
      path: `${parsed.pathname}${parsed.search || ''}`,
      method: 'GET',
      headers: {}
    };

    if (etag) {
      options.headers['If-None-Match'] = etag;
    }

    const client = parsed.protocol === 'https:' ? https : require('http');

    const request = client.request(options, (response) => {
      const { statusCode, headers } = response;
      const chunks = [];

      response.on('data', (chunk) => chunks.push(chunk));
      response.on('end', () => {
        const body = Buffer.concat(chunks).toString('utf8');
        resolve({
          statusCode,
          headers,
          body
        });
      });
    });

    request.on('error', reject);

    if (timeoutMs && Number.isFinite(timeoutMs)) {
      request.setTimeout(timeoutMs, () => {
        request.destroy(new Error('Request timed out'));
      });
    }

    request.end();
  });
}

class EventConfigFetcher {
  constructor({
    registry,
    resolveConfigPaths,
    fallbackRemote,
    logger = console,
    configFileName = 'posthog-events.json',
    onConfigApplied,
    onFetchLifecycle
  }) {
    this.registry = registry;
    this.resolveConfigPaths = resolveConfigPaths;
    this.logger = logger;
    this.configFileName = configFileName;
    this.onConfigApplied = typeof onConfigApplied === 'function' ? onConfigApplied : () => {};
    this.onFetchLifecycle = typeof onFetchLifecycle === 'function' ? onFetchLifecycle : () => {};

    this.state = {
      remote: { ...(fallbackRemote || {}) },
      configPath: null,
      fetchPromise: null,
      bootstrapPromise: null,
      timer: null,
      lastFetchAt: null,
      retryAttempt: 0,
      hadSuccessfulFetch: false
    };
  }

  async bootstrap() {
    if (this.state.bootstrapPromise) {
      return this.state.bootstrapPromise;
    }

    this.state.bootstrapPromise = (async () => {
      const loadResult = this.loadFromDisk();

      this.fetchLatest({
        force: true,
        reason: loadResult.success ? 'bootstrap_refresh' : 'bootstrap_no_cache',
        block: false
      });

      return this.registry.getSnapshot();
    })().finally(() => {
      this.state.bootstrapPromise = null;
    });

    return this.state.bootstrapPromise;
  }

  loadFromDisk() {
    const candidates = typeof this.resolveConfigPaths === 'function' ? this.resolveConfigPaths() : [];
    for (const candidate of candidates) {
      try {
        if (!fs.existsSync(candidate)) {
          continue;
        }
        const result = loadRegistryFromFile(candidate, { source: candidate });
        if (!result.success) {
          if (result.errors) {
            result.errors.forEach((message) => {
              this.logger.warn('[analytics-fetcher] Invalid config on disk:', message);
            });
          }
          continue;
        }

        this.registry.update(result.raw, { source: candidate });
        const remote = normalizeRemoteMetadata(result.raw.remote, { fallback: this.state.remote });
        this.state.remote = remote;
        this.state.configPath = candidate;
        this.onConfigApplied({
          path: candidate,
          remote,
          source: 'disk',
          snapshot: this.registry.getSnapshot()
        });
        if (result.warnings && result.warnings.length > 0) {
          result.warnings.forEach((warning) => {
            this.logger.warn('[analytics-fetcher] Config warning:', warning);
          });
        }
        return { success: true, remote };
      } catch (error) {
        this.logger.warn(
          '[analytics-fetcher] Failed to load config from disk:',
          candidate,
          error && error.message ? error.message : error
        );
      }
    }
    return { success: false };
  }

  async fetchLatest({ force = false, reason = 'manual', block = false } = {}) {
    if (this.state.timer) {
      clearTimeout(this.state.timer);
      this.state.timer = null;
    }

    if (this.state.fetchPromise) {
      return block ? this.state.fetchPromise : { success: true, pending: true };
    }

    const remote = this.state.remote || {};
    if (!remote.url) {
      this.logger.warn('[analytics-fetcher] Remote URL is not configured.');
      return { success: false, skipped: true, reason: 'missing_remote_url' };
    }

    this.state.fetchPromise = this.performFetch(remote, { reason })
      .then((result) => {
        if (result.success) {
          this.state.lastFetchAt = Date.now();
          this.state.retryAttempt = 0;
          if (result.updated) {
            this.state.hadSuccessfulFetch = true;
          }
        } else if (!result.success && !result.skipped) {
          this.scheduleRetry();
        }
        return result;
      })
      .finally(() => {
        this.state.fetchPromise = null;
      });

    if (block) {
      return this.state.fetchPromise;
    }

    this.state.fetchPromise.catch((error) => {
      this.logger.warn(
        '[analytics-fetcher] Fetch promise rejected:',
        error && error.message ? error.message : error
      );
    });

    return { success: true, pending: true };
  }

  scheduleNextFetch(delayOverrideMs) {
    if (this.state.timer) {
      clearTimeout(this.state.timer);
      this.state.timer = null;
    }

    const interval = delayOverrideMs;
    if (!interval || !Number.isFinite(interval) || interval <= 0) {
      return;
    }

    this.state.timer = setTimeout(() => {
      this.fetchLatest({ reason: 'scheduled' }).catch((error) => {
        this.logger.warn(
          '[analytics-fetcher] Scheduled fetch failed:',
          error && error.message ? error.message : error
        );
      });
    }, interval);

    if (this.state.timer && typeof this.state.timer.unref === 'function') {
      this.state.timer.unref();
    }
  }

  scheduleRetry() {
    const attempt = Math.min(this.state.retryAttempt, RETRY_DELAYS_MS.length - 1);
    const delay = RETRY_DELAYS_MS[attempt];
    this.state.retryAttempt += 1;
    this.scheduleNextFetch(delay);
  }

  async performFetch(remote, { reason }) {
    const startedAt = Date.now();
    const attempt = this.state.retryAttempt;
    try {
      this.onFetchLifecycle({
        stage: 'start',
        reason,
        remote,
        attempt
      });

      const response = await requestJson(remote.url, {
        timeoutMs: remote.timeoutMs,
        etag: remote.etag
      });

      if (response.statusCode === 304) {
        this.onFetchLifecycle({
          stage: 'not_modified',
          reason,
          remote,
          attempt,
          statusCode: response.statusCode,
          durationMs: Date.now() - startedAt
        });
        return { success: true, updated: false, skipped: true };
      }

      if (response.statusCode < 200 || response.statusCode >= 300) {
        const error = new Error(`Unexpected status code ${response.statusCode}`);
        this.onFetchLifecycle({
          stage: 'error',
          reason,
          remote,
          error,
          statusCode: response.statusCode,
          attempt,
          durationMs: Date.now() - startedAt
        });
        return { success: false, error, statusCode: response.statusCode };
      }

      let rawConfig = null;
      try {
        rawConfig = JSON.parse(response.body);
      } catch (error) {
        this.onFetchLifecycle({
          stage: 'error',
          reason,
          remote,
          error,
          attempt,
          durationMs: Date.now() - startedAt
        });
        return { success: false, error };
      }

      const normalizedRemote = normalizeRemoteMetadata(rawConfig.remote, { fallback: remote });
      const latestEtag = response.headers && response.headers.etag ? String(response.headers.etag) : null;

      rawConfig.remote = {
        ...(rawConfig.remote || {}),
        url: normalizedRemote.url,
        timeoutMs: normalizedRemote.timeoutMs,
        etag: latestEtag || normalizedRemote.etag || null
      };

      const applied = this.registry.update(rawConfig, { source: normalizedRemote.url });
      if (!applied) {
        const error = new Error('Registry rejected remote config');
        this.onFetchLifecycle({
          stage: 'error',
          reason,
          remote,
          error
        });
        return { success: false, error };
      }

      this.state.remote = {
        ...normalizedRemote,
        etag: rawConfig.remote.etag
      };
      this.persistConfig(rawConfig);

      this.onConfigApplied({
        path: this.state.configPath,
        remote: this.state.remote,
        source: 'remote',
        snapshot: this.registry.getSnapshot()
      });

      this.onFetchLifecycle({
        stage: 'success',
        reason,
        remote: this.state.remote,
        attempt,
        durationMs: Date.now() - startedAt,
        updated: true
      });

      return { success: true, updated: true };
    } catch (error) {
      this.onFetchLifecycle({
        stage: 'error',
        reason,
        remote,
        error,
        attempt,
        durationMs: Date.now() - startedAt
      });
      return { success: false, error };
    }
  }

  persistConfig(config) {
    if (!this.state.configPath) {
      const candidates = typeof this.resolveConfigPaths === 'function' ? this.resolveConfigPaths() : [];
      if (candidates.length > 0) {
        this.state.configPath = candidates[0];
      } else {
        this.state.configPath = path.join(process.cwd(), 'config', this.configFileName);
      }
    }

    try {
      fs.mkdirSync(path.dirname(this.state.configPath), { recursive: true });
      fs.writeFileSync(this.state.configPath, `${JSON.stringify(config, null, 2)}\n`, 'utf8');
    } catch (error) {
      this.logger.warn(
        '[analytics-fetcher] Failed to persist config:',
        this.state.configPath,
        error && error.message ? error.message : error
      );
    }
  }

  stop() {
    if (this.state.timer) {
      clearTimeout(this.state.timer);
      this.state.timer = null;
    }
  }
}

module.exports = {
  EventConfigFetcher
};


